#!/usr/bin/python
# -*- coding: utf-8 -*-
"""
Created on Sun Apr 13 19:00:06 2014.
Modified on Mon Nov 9 2020
@author: Vincent Ledda
"""


# Copyright (C) 2014-2020 Vincent Ledda <vledda@free.fr>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA


from sympy import (Add, Matrix, Poly, eye, init_printing, latex, ones, symbols,
                   zeros)

init_printing(use_unicode=True)
import os
import sys

latex_cmd = "rubber -m pdftex "
latex_prb = "\\documentclass[A4,12pt]{cueep}\n"

M = symbols("M")
e = symbols("e")


def ligne_tab_latex(ligne, variables):
    """Renvoie une CL au format latex.

    :param ligne: tableau de coefficients
    :param variables: tableau de variables
    :returns: str

    """

    s = ""
    var = variables[0]
    Nbvar = len(variables)
    if ligne[0] == 1:
        s = s + var
    elif ligne[0] == -1:
        s = s + "-" + var
    elif ligne[0] != 0:
        s = s + str(ligne[0]) + var

    for j in range(1, Nbvar):
        var = variables[j]

        if ligne[j] == 1:
            s = s + "+" + var
        elif ligne[j] == -1:
            s = s + "-" + var
        elif 0 < ligne[j]:
            s = s + "+" + str(ligne[j]) + var
        elif ligne[j] < 0:
            s = s + str(ligne[j]) + var
    return s


def inf_poly(a, b):
    """
	Compare le coefficient dominant de deux polynômes
    :param a:  un polynôme
    :type a: array
    :param b:  un polynôme
    :type b: array

    """
    return Poly(a, M).LC() < Poly(b, M).LC()


def __lt_(self, autre):
    return inf_poly(self, autre)


def is_neg_elem(vect):
    """Renvoie 1 si il y a un élément négatif dans le tableau

    :param vect: tableau de nombres
    :returns: int

    """
    s = 0
    for i in vect:
        if i < 0:
            s = 1
            break
    return s


class tab_simplex(object):
    """Tableau avec les méthodes de base du simplexe en calcul symbolique"""

    def __init__(self, mat_contrainte, vec_contrainte, fonc_eco):
        A = Matrix(mat_contrainte)
        self.Ncontraintes = A.shape[0]
        self.Nvaraux = 0
        self.Nvariables = A.shape[1]
        self.tableau = A.col_insert(self.Nvariables, eye(self.Ncontraintes))
        self.tableau = self.tableau.col_insert(self.NvarT, Matrix(vec_contrainte))
        eco:Matrix = Matrix(fonc_eco)
        eco=eco.transpose()

        eco = eco.col_insert(self.Nvariables, zeros(1, self.Ncontraintes + 1))
        self.tableau = self.tableau.row_insert(self.Ncontraintes, eco)

    def pivot(self, i, j):
        """Pivote avec pour pivot l'élément [i,j]
            Et renvoie la matrice élémentaire de transformation

        :param i: int
        :param j: int

        """
        if (
            self.Ncontraintes + self.Nvariables < j
            or j < 0
            or i > self.Ncontraintes
            or i < 0
        ):
            print("les indexes ne sont pas dans les bornes")
            sys.exit(1)
        elif self.tableau[i, j] != 0:
            pivot = self.tableau[i, j]
            E = eye(self.Ncontraintes + 1)
            for k in range(self.Ncontraintes + 1):
                if k == i:
                    E[k, i] = 1 / pivot
                else:
                    E[k, i] = -self.tableau[k, j] / pivot

            self.tableau = E * self.tableau
            return E

    @property
    def NvarT(self):
        """Donne le nombre total de variables"""
        return self.Nvariables + self.Ncontraintes

    def eco_max(self):
        """Donne le maximum de la dernière ligne et un indexe où ce maximum est atteint"""
        i = 0
        pos = 0

        m = self.NvarT
        n = self.Ncontraintes

        for j in range(m):
            if i < self.tableau[n, j]:
                i = self.tableau[n, j]
                pos = j
        return i, pos

    def delta_min(self, j):
        """renvoie nb si le domaine est non borné
        renvoie l'indexe de la ligne où le minimum est atteint
        renvoie l'opposé d'un indexe où le minimum est atteint (Dans le cas où il est atteint plusieurs fois)

        :param j: int
        :return i: int ou str

        """
        i = 0
        d = 0
        m = self.NvarT
        premier = 0
        delta = 0
        for k in range(self.Ncontraintes):

            if self.tableau[k, j] > 0:
                delta_current = self.tableau[k, m] / self.tableau[k, j]

                if (premier == 1) and (delta == delta_current):
                    d = 1
                if (premier == 1) and (delta > delta_current):
                    i = k
                    delta = delta_current
                if premier == 0:
                    premier = 1
                    i = k
                    delta = delta_current

        if premier == 0:
            i = "nb"
        if d == 1:
            i = -i

        return i

    @property
    def enbase(self):
        """ """
        s = []
        a = eye(self.Ncontraintes + 1)
        for j in range(self.NvarT):
            v = self.tableau[:, j]
            if v.norm() == 1:
                for k in range(self.Ncontraintes):
                    if v.dot(a[:, k]):
                        s.append(j)
        return s


class tab_simplex_generalise(tab_simplex):
    """Tableau simplexe généralisé: méthode M et méthode perturbation"""

    def __init__(self, mat_contrainte, vec_contrainte, vec_inegalite, fonc_eco):
        self.Nvaraux = 0
        super(tab_simplex_generalise, self).__init__(
            mat_contrainte, vec_contrainte, fonc_eco
        )
        self.inegalites(vec_inegalite)
        self.vectinit = (
            zeros(1, self.Nvariables)
            .col_insert(self.Nvariables, ones(1, self.Ncontraintes))
            .tolist()[0]
        )

        if not self.is_sbr(self.vectinit):
            for j in range(self.Ncontraintes):
                if vec_inegalite[j] == -1:
                    self.variable_aux(j)

    @property
    def NvarT(self):
        """Donne le nombre total de variables"""
        return self.Nvaraux + self.Nvariables + self.Ncontraintes

    def inegalites(self, vec_inegalite):
        """

        :param vec_inegalite: 

        """
        for i in range(self.Ncontraintes):
            if vec_inegalite[i] == -1:
                self.tableau[i, self.Nvariables + i] = -1

    def variable_aux(self, i):
        """

        :param i: 

        """
        C = zeros(self.Ncontraintes + 1, 1)
        C[i] = 1
        C[self.Ncontraintes] = -M

        self.tableau = self.tableau.col_insert(self.NvarT, Matrix(C))
        E = eye(self.Ncontraintes + 1)
        E[self.Ncontraintes, i] = M
        self.tableau = E * self.tableau
        self.Nvaraux += 1
        return E

    def is_sbr(self, vecteur):
        """Renvoie 1 si le vecteur représente une solution de base réalisable

        :param vecteur:

        """

        ### À reprendre
        if len(vecteur) != self.NvarT:
            print("La taille du vecteur n'est pas égale au nombre de variables")
            sys.exit(1)
        selection = []
        for j in range(self.Ncontraintes + self.Nvariables):
            if vecteur[j] == 1:
                selection.append(j)

        A = self.tableau[range(self.Ncontraintes), selection]
        B = self.tableau[
            range(self.Ncontraintes), [self.Ncontraintes + self.Nvariables]
        ]

        return 1 - is_neg_elem(A ** -1 * B)

    def eco_max(self):

        """Donne le maximum de la dernière ligne et un indexe où ce maximum est atteint"""
        i = 0
        pos = 0

        m = self.NvarT
        n = self.Ncontraintes

        for j in range(m):
            if inf_poly(i, self.tableau[n, j]):
                i = self.tableau[n, j]
                pos = j
        return i, pos


class pb_optimisation(object):
    """décrit un problème d'optimisation linéaire"""

    def __init__(
        self,
        mat_contrainte,
        vec_contrainte,
        fonc_eco,
        vec_inegalite=[],
        typepb="max",
        variables=[],
    ):
        self.mat_contrainte = mat_contrainte
        self.vec_contrainte = vec_contrainte
        self.nonborne = 0

        self.typepb = typepb
        if typepb == "min":
            self.fonc_eco = ((-1) * Matrix([fonc_eco])).tolist()[0]
        else:
            self.fonc_eco = fonc_eco

        if len(vec_inegalite) > 0:
            self.vec_inegalite = vec_inegalite
        else:
            self.vec_inegalite = ones(1, len(vec_contrainte)).tolist()[0]
        if len(variables) > 0:
            self.variables = variables
        else:
            self.variables = self.variables_default()

        self.verification()
        self.Nvariables = len(fonc_eco)
        self.Ncontraintes = len(vec_contrainte)

        if ones(1, len(vec_contrainte)).tolist()[0] == self.vec_inegalite:
            self.TS = tab_simplex(mat_contrainte, vec_contrainte, self.fonc_eco)

        else:
            self.TS = tab_simplex_generalise(
                mat_contrainte, vec_contrainte, self.vec_inegalite, self.fonc_eco
            )

    def variables_default(self):
        """ """
        liste_var = ["x", "y", "z", "t"]
        s = []
        i = len(self.fonc_eco)

        if i < 5:
            s = liste_var[0:i]
        else:
            for j in range(i):
                s.append("x_" + str(j + 1))
        return s

    def verification(self):
        """ """
        A = Matrix(self.mat_contrainte)

        if A.shape[0] != len(self.vec_contrainte):
            print(
                "Erreur: le nombre de lignes de la  matrice des contraintes n'est pas égal à la longueur du vecteur des contraintes"
            )
            sys.exit(1)
        if A.shape[1] != len(self.fonc_eco):
            print(
                "Erreur: le nombre de colonnes de la  matrice des contraintes n'est pas égal à la longueur du vecteur de la fonction éco."
            )
            sys.exit(1)
        if A.shape[1] != len(self.variables):
            print(
                "Erreur: le nombre de colonnes de la  matrice des contraintes n'est pas égal à la longueur du vecteur des variables."
            )
            sys.exit(1)
        if A.shape[0] != len(self.vec_inegalite):
            print(
                "Erreur: le nombre de lignes de la  matrice des contraintes n'est pas égal à la longueur du vecteur des inégalités."
            )
            sys.exit(1)
        if not (self.typepb == "max" or self.typepb == "min"):
            print("Erreur: le typepb doit valoir <<min>> ou <<max>>")
            sys.exit(1)

    def afficher(self, stylo):
        """Affiche la conclusion

        :param stylo:

        """
        stylo.ecrit("En donnant les valeurs: ")
        if self.typepb == "max":
            t = "maximum"
        else:
            t = "minimum"
        valeurs, maximum = self.resultat()
        for i in range(self.Nvariables):
            stylo.ecrit("%s à la variable $%s$, " % (valeurs[i], self.variables[i]))
        stylo.ecrit("on obtient un " + t + " de " + str(maximum) + ".")

    def resultat(self):
        """À modifier ne fonctionne que pour le max"""
        valeurs = []
        for i in range(self.TS.Nvariables):
            try:
                self.TS.enbase.index(i)
            except ValueError:
                valeurs.append(0)
            else:
                valeurs.append(
                    self.TS.tableau[:, i].dot(self.TS.tableau[:, self.TS.NvarT])
                )
        minmax = -self.TS.tableau[self.TS.Ncontraintes, self.TS.NvarT]
        if self.typepb == "min":
            minmax = -minmax
        return valeurs, minmax

    def resoudre(self):
        """ """
        Max, j = self.TS.eco_max()

        while inf_poly(0, Max):
            i = self.TS.delta_min(j)

            if i == "nb":
                self.nonborne = 1
                break
            if i < 0:
                ## pour l'instant... ecrire methode de perturbation pour éviter le cyclage
                i = -i

            self.TS.pivot(i, j)

            Max, j = self.TS.eco_max()
        if self.nonborne == 0:
            print(self.TS.tableau)
        else:
            print("Le domaine n'est pas borné")

    def resoudre_pas_a_pas(self, fichier=""):
        """

        :param fichier: Default value = "")

        """
        stylo = ecrivain(fichier)
        stylo.export_latex_init(
            self.TS, self.variables, self.fonc_eco, self.vec_inegalite, self.typepb
        )
        Max, j = self.TS.eco_max()
        while inf_poly(0, Max):
            i = self.TS.delta_min(j)
            if i == "nb":

                self.nonborne = 1
                break
            elif i < 0:
                ## pour l'instant... à faire méthode de perturbation pour éviter le cyclage
                i = -i
            E = self.TS.pivot(i, j)
            stylo.export_latex(self.TS.tableau, E)

            Max, j = self.TS.eco_max()

        if self.nonborne == 0:
            self.afficher(stylo)
        else:
            stylo.ecrit("Le domaine n'est pas borné. \n")
            if self.typepb == "max" and inf_poly(0, Max):
                    stylo.ecrit("Max E=$+\\infty$")
            elif self.typepb == "min" and inf_poly(0, Max):
                    stylo.ecrit("Min E=$-\\infty$")

        stylo.fin()

        if fichier != "":
            stylo.compile_()


class ecrivain(object):
    """ """
    def __init__(self, fichier=""):
        self.fichier = fichier
        if fichier == "":
            self.sortie = sys.stdout
        else:
            self.sortie = open(fichier, "w")
        self.sortie.write(latex_prb + "\\begin{document}\n")

    def fin(self):
        """ """
        self.sortie.write("\\end{document}")
        self.sortie.close()

    def elem_ligne(self, E):
        """Renvoie la traduction <<symbolique>> d'une matrice de transformation élémentaire sur les lignes

        :param E:

        """
        n = E.shape[0]
        F = E - eye(n)
        j = 1
        for i in range(n):
            if F[:, i].norm() > 0:
                j = i

        texte = []

        for i in range(n):
            if i == j:
                texte.append(
                    "L_{%s}\leftarrow {%s}L_{%s}" % (i + 1, latex(E[i, j]), i + 1)
                )
            else:
                if E[i, j] == 0:
                    texte.append("L_{%s}\leftarrow L_{%s}" % (i + 1, i + 1))

                elif type(E[i, j]) is Add:
                    texte.append(
                        "L_{%s}\leftarrow L_{%s}+{%s}L_{%s}"
                        % (i + 1, i + 1, "(" + latex(E[i, j]) + ")", j + 1)
                    )
                elif str(E[i, j]) < str(0):
                    texte.append(
                        "L_{%s}\leftarrow L_{%s}{%s}L_{%s}"
                        % (i + 1, i + 1, latex(E[i, j]), j + 1)
                    )
                else:
                    texte.append(
                        "L_{%s}\leftarrow L_{%s}+{%s}L_{%s}"
                        % (i + 1, i + 1, latex(E[i, j]), j + 1)
                    )
        return texte

    def compile_(self):
        """ """
        if self.fichier != "":

            os.system(latex_cmd + " " + self.fichier)

    def ecrit(self, texte):
        """

        :param texte:

        """
        self.sortie.write(texte)

    def export_latex_init(self, tableau, variables, fonc_eco, inegalites, typepb):
        """

        :param tableau: param variables:
        :param fonc_eco: param inegalites:
        :param typepb: param variables:
        :param fonc_eco: param inegalites:
        :param typepb: param variables:
        :param inegalites: param typepb:
        :param variables: param inegalites:
        :param typepb:

        """
        self.sortie.write("\\begin{eqnarray*}\\left\\lbrace\\begin{array}{rl}\n")
        for i in range(tableau.Ncontraintes):
            ligne = ligne_tab_latex(tableau.tableau[i, :], variables)

            if inegalites[i] == 1:
                ligne = (
                    ligne + "&\\leq" + str(tableau.tableau[i, tableau.NvarT]) + "\\\\\n"
                )
            else:
                ligne = (
                    ligne + "&\\geq" + str(tableau.tableau[i, tableau.NvarT]) + "\\\\\n"
                )
            self.sortie.write(ligne)

        self.sortie.write("\\end{array}\\right.\\\\\n")
        if typepb == "max":
            ligne = "\\text{Max E=}"
            ligne = ligne + ligne_tab_latex(fonc_eco, variables)
        else:
            ligne = "\\text{Min E=}"
            ligne = ligne + ligne_tab_latex(
                ((-1) * Matrix([fonc_eco])).tolist()[0], variables
            )
        self.sortie.write(ligne)
        self.sortie.write(
            "\\end{eqnarray*}\\begin{equation*}\n%s\\end{equation*}\n"
            % (latex(tableau.tableau))
        )

    def export_latex(self, tableau, E):
        """

        :param tableau: param E:
        :param E:

        """
        self.sortie.write("\\begin{equation*}\n\\begin{array}[h]{l}\n")
        for ligne in self.elem_ligne(E):
            self.sortie.write(ligne + "\\\\")
        self.sortie.write("\\end{array}%s\\end{equation*}\n" % (latex(tableau)))


"""
A=pb_optimisation([[1,1],[-2,3],[2,-1]],[14,12,12],[1,3])
A.resoudre_pas_a_pas("simplexe_exemple_1.tex")


B=pb_optimisation([[2,-1],[1,-1],[1,1]],[-2,2,5],[-1,1],[-1,1,1],"min")
B.resoudre_pas_a_pas("simplexe_exemple_2.tex")


C=pb_optimisation([[1,1],[1,0],[0,1]],[6,4,3],[5,7],[-1,-1,1])
C.resoudre_pas_a_pas("simplexe_exemple_3.tex")

D=pb_optimisation([[-2,1],[-1,2],[1,-4]],[2,5,4],[1,2])
D.resoudre_pas_a_pas("simplexe_exemple_4.tex")





#Pb avec solution dégénérée
E=pb_optimisation([[1,-1,1],[2,1,4],[5,-1,1]],[3,8,5],[14,0,10])
E.resoudre_pas_a_pas("simplexe_exemple_5.tex")
E.TS.pivot(2,3)
print(E.TS.tableau)



F=pb_optimisation([[1,4,0,-1,3],[-1,1,1,0,3],[0,1,-2,0,2],[1,0,0,1,5]],[8,8,4,8],[1,-4,1,5,1])
F.resoudre_pas_a_pas("simplexe_exemple_6.tex")


G=pb_optimisation([[1,1,1],[10,20,6]],[50,400],[600,1000,720],[1,1],"max",["t","l","r"])
G.resoudre_pas_a_pas("simplexe_exemple_7.tex")


H=pb_optimisation([[5,7,12],[2,10,4]],[420,360],[5,7,6],[1,1],"max",["a","b","c"])
H.resoudre_pas_a_pas("simplexe_exemple_8.tex")

I=pb_optimisation([[1,2,1],[2,3,3],[2,1,1]],[12,20,14],[12,20,8],[1,1,1],"max")
I.resoudre_pas_a_pas("simplexe_exemple_9.tex")

"""

